/**
 * @module CLI
 */

import { ProjectFileCoder } from "./ProjectFileCoder.mjs";
import path from "node:path";

/**
 * Prints CLI usage instructions.
 *
 * @returns {void}
 */
function printUsage() {
  console.log(
    `
Usage:
  node cli.mjs <project_path> <code|decode>

Examples:
  node cli.mjs ./my-project code
  node cli.mjs ./my-project decode
`.trim(),
  );
}

/**
 * Entry point for CLI execution.
 *
 * Expected arguments:
 * - project path
 * - operation: "code" or "decode"
 */
const [, , projectPath, operation] = process.argv;

if (!projectPath || !operation) {
  printUsage();
  process.exit(1);
}

const op = operation.toLowerCase();

if (op !== "code" && op !== "decode") {
  console.error(`Unknown operation: ${operation}`);
  printUsage();
  process.exit(1);
}

const coder = new ProjectFileCoder();

try {
  const abs = path.resolve(projectPath);

  if (op === "code") {
    await coder.encodeFiles(abs);
    console.log(`Encoding completed: ${abs}`);
  } else {
    await coder.decodeFiles(abs);
    console.log(`Decoding completed: ${abs}`);
  }
} catch (error) {
  console.error("Error:", error?.message ?? error);
  process.exit(1);
}
