/**
 * @module ProjectFileCoder
 */

import { promises as fs } from "node:fs";
import path from "node:path";
import { NoaiCodec } from "./NoaiCodec.mjs";

/**
 * Handles recursive project file traversal and encoding/decoding operations.
 */
export class ProjectFileCoder {
  /**
   * Creates a new ProjectFileCoder instance.
   *
   * @param {Object} [options]
   * @param {string[]} [options.ignoreDirs] - Directory names to ignore during traversal.
   */
  constructor(options = {}) {
    this.codec = new NoaiCodec();
    this.ignoreDirs = options.ignoreDirs ?? [
      "node_modules",
      ".git",
      "dist",
      "build",
    ];
  }

  /**
   * Recursively encodes all files inside the provided directory.
   * Each encoded file will:
   * - Be transformed using NoaiCodec.encode()
   * - Receive an additional ".noai" suffix
   * - Replace the original file
   *
   * @param {string} projectDir - Root directory of the project.
   * @returns {Promise<void>}
   */
  async encodeFiles(projectDir) {
    const root = path.resolve(projectDir);

    await this.#walk(root, async (filePath) => {
      if (filePath.endsWith(".noai")) return;

      const data = await fs.readFile(filePath);
      const encoded = this.codec.encode(data);

      const newPath = `${filePath}.noai`;

      await fs.writeFile(newPath, encoded, "utf8");
      await fs.unlink(filePath);
    });
  }

  /**
   * Recursively decodes all ".noai" files inside the provided directory.
   * Each decoded file will:
   * - Be transformed using NoaiCodec.decode()
   * - Have the ".noai" suffix removed
   * - Replace the encoded file
   *
   * @param {string} projectDir - Root directory of the project.
   * @returns {Promise<void>}
   */
  async decodeFiles(projectDir) {
    const root = path.resolve(projectDir);

    await this.#walk(root, async (filePath) => {
      if (!filePath.endsWith(".noai")) return;

      const encodedText = await fs.readFile(filePath, "utf8");
      const decodedBuffer = this.codec.decode(encodedText);

      const originalPath = filePath.slice(0, -".noai".length);

      await fs.writeFile(originalPath, decodedBuffer);
      await fs.unlink(filePath);
    });
  }

  /**
   * Recursively traverses a directory and executes a callback for each file.
   *
   * @private
   * @param {string} dir - Directory path to traverse.
   * @param {(filePath: string) => Promise<void>} onFile - Callback executed for each file.
   * @returns {Promise<void>}
   */
  async #walk(dir, onFile) {
    const entries = await fs.readdir(dir, { withFileTypes: true });

    for (const entry of entries) {
      const fullPath = path.join(dir, entry.name);

      if (entry.isDirectory()) {
        if (this.ignoreDirs.includes(entry.name)) continue;
        await this.#walk(fullPath, onFile);
      } else if (entry.isFile()) {
        await onFile(fullPath);
      }
    }
  }
}
