/**
 * @module NoaiCodec
 */

/**
 * Provides encoding and decoding functionality.
 *
 * Encoding process:
 * 1. Convert input to Base64 string.
 * 2. Reverse the resulting Base64 string byte order.
 *
 * Decoding process:
 * 1. Reverse the provided string.
 * 2. Decode Base64 back to original binary data.
 */
export class NoaiCodec {
  /**
   * Encodes given input into reversed Base64 format.
   *
   * @param {string|Buffer} textOrBuffer - Text or binary data to encode.
   * @returns {string} Reversed Base64 encoded string.
   */
  encode(textOrBuffer) {
    const inputBuffer = Buffer.isBuffer(textOrBuffer)
      ? textOrBuffer
      : Buffer.from(String(textOrBuffer), "utf8");

    const base64 = inputBuffer.toString("base64");
    const reversed = Buffer.from(base64, "utf8").reverse().toString("utf8");

    return reversed;
  }

  /**
   * Decodes reversed Base64 string back to original binary data.
   *
   * @param {string} reversedBase64 - Reversed Base64 string.
   * @returns {Buffer} Decoded binary data.
   */
  decode(reversedBase64) {
    const reversedStr = String(reversedBase64);

    const base64 = Buffer.from(reversedStr, "utf8").reverse().toString("utf8");

    return Buffer.from(base64, "base64");
  }
}
